from qgis.PyQt.QtCore import QThread, pyqtSignal, QMutex, QMutexLocker
import asyncio
import time
from bleak import BleakClient, BleakScanner, BleakGATTCharacteristic
from qgis.core import QgsMessageLog, Qgis


TIME_LIMIT = 900 # s
MODEL_FEE1_UUID = "0000fee1-0000-1000-8000-00805f9b34fb" # value for BLE measurement


class BleReaderThread(QThread):
    # Definisce i segnali di errore e di misurazione
    measurement_received = pyqtSignal(str)
    connection_status = pyqtSignal(bool)  # Stato connessione
    connection_timeout = pyqtSignal(str)
    error_signal = pyqtSignal(str)



    def __init__(self, DEVICE_ADRESS, DEVICE_NAME, MODEL_FEE1_UUID=MODEL_FEE1_UUID):
        # Inizializza il processo BleReaderThread
        super().__init__()
        self._mutex = QMutex()
        self._abort = False
        self._device_name = DEVICE_NAME
        self._device_address = DEVICE_ADRESS
        self._characteristic_uuid = MODEL_FEE1_UUID
        # Inizializza il client BLE come None
        self._client = None
        self._start_time = None
        self._connection_lost = False

        # Imposta il flag di esecuzione a True
        # self.is_running = True

        self.arr = []
        self.signal_cnt = None

    def abort(self):
        # print("4) Chiamaata funzione abort del segnale")
        with QMutexLocker(self._mutex):
            self._abort = True
            # print(f"{'E'*20} (self.abort()) Richiesta interruzione connessione BLE")
            QgsMessageLog.logMessage("Richiesta interruzione connessione BLE", "BLEReader", Qgis.Info)

    def _check_abort(self):
        with QMutexLocker(self._mutex):
            return self._abort

    def notification_handler(self, sender: BleakGATTCharacteristic, data: bytearray):
        # try: # (caso di stringa 40 char, si pianta tutto se uso try except, così invece non sente solo il segnale)
        s = data.decode()

        # se finisce per \n allora ho chiuso trasmissione dati e passo all'eleborazione
        # if \n is the last character of the string, then I have finished transmitting data and move on to processing
        if s and s.endswith('\n'):

            #################################################################################
            #################################################################################
            s = s.replace('\n', '').replace('\t', '')
            self.arr.append(s)
            #################################################################################
            #################################################################################


            x = ''.join(self.arr).strip() # x = $XRPT,HV,1.5,-0.4,1.6,M,-26.3,D,226.9,D,*L

            if len(x.split('*')) > 1:
                # il dispositivo passa il checum
                # come prima cosa recupero il checksum che mi passa il dispositivo
                received_char = x.split('*')[1]
                received_ascii = ord(received_char)
                received_hex = hex(received_ascii)[2:].upper()

                x = x.split('*')[0] # x = $XRPT,HV,1.5,-0.4,1.6,M,-26.3,D,226.9,D,

                # Se la stringa inizia con un $ allora la sintassi è corretta e quindi procedo
                x = x[1:] # x = XRPT,HV,1.5,-0.4,1.6,M,-26.3,D,226.9,D,
                # #-->print(f"x -------------------------> {x}")

                calculated_ascii = 0
                for s in x:
                    # La linea sotto con ord() calcola il valore ASCII del carattere s e lo utilizza per aggiornare il valore di calculated_ascii tramite un'operazione XOR.
                    calculated_ascii ^= ord(s)

                # la trasformo in stringa e tolgo l'indicatore di hex
                calculated_hex = str(hex(calculated_ascii)).replace("0x", "").upper() # calculated_hex = 4C
                calculated_char = chr(calculated_ascii)

                # #-->print(f"received_char ----> {received_char}") # received_char = L
                # #-->print(f"received_ascii ----> {received_ascii}")
                # #-->print(f"received_hex ----> {received_hex}")
                #
                # #-->print(f"calculated_ascii ----> {calculated_ascii}")
                # #-->print(f"calculated_hex ----> {calculated_hex}")
                # #-->print(f"calculated_char ----> {calculated_char}")

                # per test errore
                # if calculated_ascii >= 75:
                #     calculated_ascii += 1


                check_char = received_char == calculated_char
                check_ascii = received_ascii == calculated_ascii
                check_hex = received_hex == calculated_hex
                all_checked = check_ascii and check_char and check_hex

                # if True or received_char.upper() == calculated_hex:
                if all_checked:
                    # #-->print("all checked ;)")
                    # Il checksun calcolato e quello passato dal dispositivo coincido e dunque il dato è buono
                    self.signal_cnt = "{}@{}@{}".format(x.split(',')[8], x.split(',')[4], x.split(',')[6])
                    # #-->print(f"signal_cnt ------------> {self.signal_cnt}")

                    self.measurement_received.emit(self.signal_cnt)

                else:
                    print("Error with Signal Transmission.")
                    print(f"check_char -------------------> {check_char}")
                    print(f"check_ascii -------------------> {check_ascii}")
                    print(f"check_hex -------------------> {check_hex}")

            self.arr = []
        elif s and s.startswith('$'):
            # se parte con dollaro inizia il segnale, inizializzo la lista
            self.arr = [s]
        elif s:
            # se non è vuota e non inizia con dollaro, continuo a riempire la stringa
            self.arr.append(s)


        # except Exception as e:
        #     #-->print(e)
        #     self.error_signal.emit(f"Errore nella gestione dei dati: {str(e)}")

    async def run_ble_client(self):
        # global signal_cnt

        # Stampa un messaggio di connessione al BT
        #-->print("\n---> BLE device connection...")
        try:
            async with BleakClient(self._device_address) as client:
                print(f"\n------------- Device will be disconnect after {TIME_LIMIT} seconds -------------")
                #-->print("\n---> Recording data...")
                self._client = client
                start_time = time.time()
                self.connection_status.emit(True)
                QgsMessageLog.logMessage("Connesso al dispositivo BLE", "BLEReader", Qgis.Info)
                # attiva le notifiche per il client
                await client.start_notify(
                    self._characteristic_uuid,
                    self.notification_handler
                )

                # Continua a eseguire finché il flag di esecuzione è True
                # loop principale
                #-->print("T"*300)
                #-->print(TIME_LIMIT)
                while not self._check_abort(): # self.is_running:
                    
                    await asyncio.sleep(0.1)
                    # print(".")
                    # #-->print(f"Tempo che passa -----------------------------------> {time.time() - start_time}")
                    # print(f"{time.time() - start_time}: ({time.time() - start_time > TIME_LIMIT})")


                    if time.time() - start_time > TIME_LIMIT:
                        self._connection_lost = True

                        self.abort()
                        self.connection_timeout.emit(str(f"TIMEOUT: connection time of {round(TIME_LIMIT/60, 2)} minutes exceeded."))
                        break

                    if not client.is_connected:
                        self._connection_lost = True
                        break



                await client.stop_notify(self._characteristic_uuid)

                # Disconnette il client BLE
                # await self._client.disconnect()

        except Exception as e:
            # print(e)
            self.error_signal.emit(f"Errore di connessione: {str(e)}")

        finally:
            # print("5) arrivo al finally")
            self.connection_status.emit(False)
            self._client = None

    def run(self):
        # Chiamata alla funzione asincrona per l'esecuzione del BLE
        # #-->print("\n---> run_ble_client()...")
        # asyncio.run(self.run_ble_client())
        try:
            # Crea un nuovo event loop per asyncio
            loop = asyncio.new_event_loop()
            asyncio.set_event_loop(loop)

            # Esegui il client BLE
            loop.run_until_complete(self.run_ble_client())

            # Chiudi l'event loop
            loop.close()

        except Exception as e:
            self.error_signal.emit(f"Errore critico: {str(e)}")
        finally:
            self.cleanup()

    def cleanup(self):
        with QMutexLocker(self._mutex):
            self._abort = False
            self._start_time = None
        QgsMessageLog.logMessage("Pulizia thread BLE completata", "BLEReader", Qgis.Info)

















































