import asyncio
import bleak
from qgis.PyQt.QtCore import QThread, pyqtSignal, QMutex, QMutexLocker
from qgis.core import QgsMessageLog, Qgis

SCANNIG_TIME = 8 # seconds

class BluetoothScannerThread(QThread):
    # signal --> [dev_name: dev_address] to ui thread
    devices_found = pyqtSignal(list)
    error_signal = pyqtSignal(str)

    def __init__(self):
        super().__init__()

        self._abort = False
        self._mutex = QMutex()
        self._devices_list = []

    def abort(self):
        with QMutexLocker(self._mutex):
            self._abort = True
            QgsMessageLog.logMessage("Richiesta interruzione scansione Bluetooth", "BluetoothScanner", Qgis.Info)


    async def scan_devices(self):
        try:
            # _devices_list = []
            #-->print(1)
            scanner = bleak.BleakScanner()
            devices = await scanner.discover(
                timeout=SCANNIG_TIME
                # return_adv=True
            )
            #-->print("^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^")
            #-->print(devices)
            return devices

        except Exception as e:
            self.error_signal.emit(f"Errore durante la scansione: {str(e)}")
            return []


    def run(self):
        # run when in ui thread start() is called

        try:
            QgsMessageLog.logMessage("Start Bluetooth LE Scanning", "BluetoothScanner", Qgis.Info)

            # async loop for scanning
            #-->print("^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^")
            loop = asyncio.new_event_loop()
            asyncio.set_event_loop(loop)
            devices = loop.run_until_complete(self.scan_devices())
            #-->print("^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^")
            #-->print(devices)
            for device in devices:
                with QMutexLocker(self._mutex):
                    if self._abort:
                        break

                # print(f"\t\t---> {device.name}: {device.address}")
                if device.name:
                    self._devices_list.append({device.name: device.address})

            # Emetti il segnale con devices trovati
            if not self._abort:
                self.devices_found.emit(self._devices_list)

            # chiudi l'event loop
            loop.close()

        except Exception as e:
            self.error_signal.emit(f"Errore durante la scansione: {str(e)}")
            QgsMessageLog.logMessage(f"Errore scansione Bluetooth: {str(e)}", "BluetoothScanner", Qgis.Critical)
        finally:

            #-->print(f"{self._devices_list = }")
            #-->print(f"{self._abort = }")
            self.cleanup()

    def cleanup(self):
        with QMutexLocker(self._mutex):
            self._devices_list.clear()
            self._abort = False


        #-->print(f"{self._devices_list = }")
        #-->print(f"{self._abort = }")

        QgsMessageLog.logMessage("Pulizia thread Bluetooth completata", "BluetoothScanner", Qgis.Info)




